#!/usr/bin/env python

#
# I simulate a PMAC with the exphi PLC running.
#
# For usage, see my printUsage function or run me with my -h option.
#

import getopt
import asynchat
import asyncore
import os
import re
import signal
import socket
import sys
import time

DEFAULT_PORT = 50436

def main(args):
  (port, isDebugOutputEnabled) = parseCommandLineArgs(args)
  d = ConnectionDispatcher(port, isDebugOutputEnabled)
  asyncore.loop()

def getProgramName(args=None):
  if args == None: args = sys.argv
  if len(args) == 0 or args[0] == "-c": return "PROGRAM_NAME"
  return os.path.basename(args[0])

def initializeSignalHandlers():
  signalToNameMap = {
    signal.SIGHUP:"SIGHUP",
    signal.SIGINT:"SIGINT",
    signal.SIGPIPE:"SIGPIPE",
    signal.SIGALRM:"SIGALRM",
    signal.SIGTERM:"SIGTERM",
    signal.SIGUSR1:"SIGUSR1",
    signal.SIGUSR2:"SIGUSR2"
  }

  def signalHandler(signum, frame):
    if signum in [signal.SIGINT, signal.SIGTERM]:
      print
      print "%s caught. Exiting." % signalToNameMap[signum]
      sys.exit(0)
    else:
      print >> sys.stderr
      print >> sys.stderr, "%s caught. Exiting." % signalToNameMap[signum]
      sys.exit(1)

  for each in signalToNameMap.keys():
    signal.signal(each, signalHandler)

def parseCommandLineArgs(args):
  (options, extra) = getopt.getopt(args[1:], "dp:h", ["debug", "port=",
    "help"])

  port = DEFAULT_PORT
  isDebugOutputEnabled = False

  for eachOptName, eachOptValue in options:
    if eachOptName in ("-d", "--debug"):
      isDebugOutputEnabled = True
    elif eachOptName in ("-p", "--port"):
      port = int(eachOptValue)
    elif eachOptName in ("-h", "--help"):
      printUsage(sys.stdout)
      sys.exit(0)

  if len(extra) > 0:
    print >> sys.stderr, "Error: unexpected command line argument \"%s\"" % \
      extra[0]
    printUsage(sys.stderr)
    sys.exit(1)

  return (port, isDebugOutputEnabled)

def printUsage(outStream):
  print >> outStream, """\
Usage: %s [-dph]

Options:
  -d,--debug        Print debug messages to stderr
  -p,--port=NUMBER  Listen on the specified port NUMBER for incoming
                    connections (default: %d)
  -h,--help         Print usage message and exit\
""" % (getProgramName(), DEFAULT_PORT)


class ConnectionDispatcher(asyncore.dispatcher):
  def __init__(self, port, debugOutputEnabled=False):
    asyncore.dispatcher.__init__(self)
    self.port = port
    self.debugOutputEnabled = debugOutputEnabled
    self.pmac = Pmac()
    self.create_socket(socket.AF_INET, socket.SOCK_STREAM)
    self.set_reuse_addr()
    self.bind(("", port))
    self.listen(5)

  def handle_accept(self):
    ConnectionSession(self.accept(), self.pmac, self.debugOutputEnabled)


class ConnectionSession(asynchat.async_chat):
  PMAC_P1 = "P1"
  PMAC_P2 = "P2"
  PMAC_P3 = "P3"
  PMAC_P4 = "P4"
  PMAC_P5 = "P5"
  PMAC_P6 = "P6"
  PMAC_P7 = "P7"
  PMAC_P8 = "P8"

  SetPhiPosRe = re.compile(r"PhiPos=(?P<val>[0-9-]+)$", re.I)

  GetArmedRe = re.compile(PMAC_P1 + r"$", re.I)
  SetArmedRe = re.compile(PMAC_P1 + r"=(?P<val>[01])$", re.I)

  GetOpenPosRe = re.compile(PMAC_P2 + r"$", re.I)
  SetOpenPosRe = re.compile(PMAC_P2 + r"=(?P<val>[0-9-]+)$", re.I)

  GetClosePosRe = re.compile(PMAC_P3 + r"$", re.I)
  SetClosePosRe = re.compile(PMAC_P3 + r"=(?P<val>[0-9-]+)$", re.I)

  GetExpTrigEnaRe = re.compile(PMAC_P4 + r"$", re.I)
  SetExpTrigEnaRe = re.compile(PMAC_P4 + r"=(?P<val>[01])$", re.I)

  GetExpTrigPosRe = re.compile(PMAC_P5 + r"$", re.I)
  SetExpTrigPosRe = re.compile(PMAC_P5 + r"=(?P<val>[0-9-]+)$", re.I)

  GetMaxExpTimeRe = re.compile(PMAC_P6 + r"$", re.I)
  SetMaxExpTimeRe = re.compile(PMAC_P6 + r"=(?P<val>\d+)$", re.I)

  GetShutterRe = re.compile(PMAC_P7 + r"$", re.I)
  SetShutterRe = re.compile(PMAC_P7 + r"=(?P<val>[01])$", re.I)

  GetExpTrigRe = re.compile(PMAC_P8 + r"$", re.I)
  SetExpTrigRe = re.compile(PMAC_P8 + r"=(?P<val>[01])$", re.I)

  def __init__(self, (conn, addr), pmac, debugOutputEnabled=False):
    asynchat.async_chat.__init__(self, conn)
    self.debugOutputEnabled = debugOutputEnabled
    self.set_terminator("\r")
    self.outputTerminator = "\006"
    self.pmac = pmac
    self.buffer = ""

  def collect_incoming_data(self, data):
    self.buffer = self.buffer + data

  def found_terminator(self):
    data = self.buffer
    self.buffer = ""
    if self.debugOutputEnabled:
      print >> sys.stderr, "< \"%s\"" % \
        (data + self.get_terminator()).encode("string_escape")
    self.handleClientRequest(data)

  def handleClientRequest(self, request):
    request = request.strip()
    match = self.GetOpenPosRe.match(request)
    if match != None:
      self.sendClientResponse(str(self.pmac.getOpenPos()))
      return
    match = self.GetClosePosRe.match(request)
    if match != None:
      self.sendClientResponse(str(self.pmac.getClosePos()))
      return
    match = self.GetExpTrigEnaRe.match(request)
    if match != None:
      self.sendClientResponse(str(self.pmac.getExpTrigEna()))
      return
    match = self.GetExpTrigPosRe.match(request)
    if match != None:
      self.sendClientResponse(str(self.pmac.getExpTrigPos()))
      return
    match = self.GetArmedRe.match(request)
    if match != None:
      self.sendClientResponse(str(self.pmac.getArmed()))
      return
    match = self.GetMaxExpTimeRe.match(request)
    if match != None:
      self.sendClientResponse(str(self.pmac.getMaxExpTime()))
      return
    match = self.GetShutterRe.match(request)
    if match != None:
      self.sendClientResponse(str(self.pmac.getShutter()))
      return
    match = self.GetExpTrigRe.match(request)
    if match != None:
      self.sendClientResponse(str(self.pmac.getExpTrig()))
      return
    match = self.SetPhiPosRe.match(request)
    if match != None:
      self.pmac.setPhiPos(long(match.group("val")))
      self.sendClientResponse()
      return
    match = self.SetArmedRe.match(request)
    if match != None:
      self.pmac.setArmed(int(match.group("val")))
      self.sendClientResponse()
      return
    match = self.SetShutterRe.match(request)
    if match != None:
      self.pmac.setShutter(int(match.group("val")))
      self.sendClientResponse()
      return
    match = self.SetExpTrigRe.match(request)
    if match != None:
      self.pmac.setExpTrig(int(match.group("val")))
      self.sendClientResponse()
      return
    match = self.SetOpenPosRe.match(request)
    if match != None:
      self.pmac.setOpenPos(long(match.group("val")))
      self.sendClientResponse()
      return
    match = self.SetClosePosRe.match(request)
    if match != None:
      self.pmac.setClosePos(long(match.group("val")))
      self.sendClientResponse()
      return
    match = self.SetExpTrigEnaRe.match(request)
    if match != None:
      self.pmac.setExpTrigEna(int(match.group("val")))
      self.sendClientResponse()
      return
    match = self.SetExpTrigPosRe.match(request)
    if match != None:
      self.pmac.setExpTrigPos(long(match.group("val")))
      self.sendClientResponse()
      return
    match = self.SetMaxExpTimeRe.match(request)
    if match != None:
      self.pmac.setMaxExpTime(long(match.group("val")))
      self.sendClientResponse()
      return

    print >> sys.stderr, "error: unknown command \"%s\"" % \
      request.encode("string_escape")

  def sendClientResponse(self, response=""):
    data = response + self.outputTerminator
    self.push(data)
    if self.debugOutputEnabled:
      print >> sys.stderr, "> \"%s\"" % data.encode("string_escape")


class Pmac:
  ExpTrigPulseWidth = 1.0 # much larger than normal so human can see it change

  def __init__(self):
    self.phiPos = 0
    self.armed = 0
    self.openPos = 0
    self.closePos = 0
    self.expTrigEna = 0
    self.expTrigPos = 0
    self.maxExpTime = 0.0
    self.shutterCloseTime = 0.0
    self.shutter = 0
    self.expTrig = 0
    self.expTrigLowTime = 0.0
    self.triggeredExposure = 0

  def setPhiPos(self, position):
    self.phiPos = position
    self.evaluatePlc()

  def getArmed(self):
    return self.armed

  def setArmed(self, armed):
    if self.armed == armed: return
    if not self.armed:
      if self.closePos >= self.openPos:
        if self.phiPos >= self.openPos: return
        if self.expTrigEna and self.phiPos >= self.expTrigPos: return
      else:
        if self.phiPos <= self.openPos: return
        if self.expTrigEna and self.phiPos <= self.expTrigPos: return
    self.armed = armed
    if armed: self.triggeredExposure = 0
    self.evaluatePlc()

  def getShutter(self):
    self.evaluatePlc()
    return self.shutter

  def setShutter(self, shutter):
    self.shutter = shutter
    if shutter == 1:
      self.shutterCloseTime = time.time() + self.maxExpTime

  def getExpTrig(self):
    self.evaluatePlc()
    return self.expTrig

  def setExpTrig(self, expTrig):
    self.expTrig = expTrig
    if expTrig == 1:
      self.expTrigLowTime = time.time() + self.ExpTrigPulseWidth

  def getOpenPos(self):
    return self.openPos

  def setOpenPos(self, position):
    self.openPos = position
    self.evaluatePlc()

  def getClosePos(self):
    return self.closePos

  def setClosePos(self, position):
    self.closePos = position
    self.evaluatePlc()

  def getExpTrigEna(self):
    return self.expTrigEna

  def setExpTrigEna(self, enabled):
    self.expTrigEna = enabled

  def getExpTrigPos(self):
    return self.expTrigPos

  def setExpTrigPos(self, position):
    self.expTrigPos = position
    self.evaluatePlc()

  def getMaxExpTime(self):
    return int(self.maxExpTime * 1000.0)

  def setMaxExpTime(self, maxExpTimeInMsec):
    self.maxExpTime = float(maxExpTimeInMsec) / 1000.0
    self.evaluatePlc()

  def evaluatePlc(self):
    if self.shutter == 1 and self.maxExpTime != 0.0 and \
        time.time() > self.shutterCloseTime:
      self.shutter = 0
      self.armed = 0
    if self.expTrig == 1 and time.time() > self.expTrigLowTime:
      self.expTrig = 0
    if not self.armed: return
    if self.closePos >= self.openPos:
      if self.phiPos >= self.openPos and self.phiPos <= self.closePos:
        if self.shutter == 0:
          self.shutter = 1
          self.shutterCloseTime = time.time() + self.maxExpTime
      if self.expTrigEna and self.phiPos >= self.expTrigPos:
        if self.expTrig == 0 and not self.triggeredExposure:
          self.expTrig = 1
          self.expTrigLowTime = time.time() + self.ExpTrigPulseWidth
          self.triggeredExposure = 1
      if self.phiPos >= self.closePos:
        self.shutter = 0
        self.armed = 0
    else:
      if self.phiPos <= self.openPos and self.phiPos >= self.closePos:
        if self.shutter == 0:
          self.shutter = 1
          self.shutterCloseTime = time.time() + self.maxExpTime
      if self.expTrigEna and self.phiPos <= self.expTrigPos:
        if self.expTrig == 0 and not self.triggeredExposure:
          self.expTrig = 1
          self.expTrigLowTime = time.time() + self.ExpTrigPulseWidth
          self.triggeredExposure = 1
      if self.phiPos <= self.closePos:
        self.shutter = 0
        self.armed = 0


if __name__ == '__main__':
  try:
    initializeSignalHandlers()
    main(sys.argv)
  except Exception, e:
    if isinstance(e, SystemExit):
      raise e
    else:
      print >> sys.stderr
      print >> sys.stderr, "Error: %s" % e
      sys.exit(1)
