#!/usr/bin/env python

#
# I establish a TCP/IP connection to the PMAC simulation server and issue
# position update commands for each new position reported on STDIN.
#
# The format of the STDIN data is either lines like camonitor produces:
#
#   ioc23:m1.RRBV  2009-07-20  11:05:11.486447  82
#
# or lines containing just one integer:
#
#   82
#
# For the camonitor format, only the rightmost column is used.  For both
# formats, the rightmost column is an integer representing the position of the
# Phi motor in counts.
#
# For usage, see my printUsage function or run me with my -h option.
#

import getopt
import os
import signal
import socket
import sys

DEFAULT_PORT = 50436

def main(args):
  (hostname, port, isDebugOutputEnabled) = parseCommandLineArgs(args)
  s = ConnectionSession(hostname, port, isDebugOutputEnabled)
  s.go()

def getProgramName(args=None):
  if args == None: args = sys.argv
  if len(args) == 0 or args[0] == "-c": return "PROGRAM_NAME"
  return os.path.basename(args[0])

def initializeSignalHandlers():
  signalToNameMap = {
    signal.SIGHUP:"SIGHUP",
    signal.SIGINT:"SIGINT",
    signal.SIGPIPE:"SIGPIPE",
    signal.SIGALRM:"SIGALRM",
    signal.SIGTERM:"SIGTERM",
    signal.SIGUSR1:"SIGUSR1",
    signal.SIGUSR2:"SIGUSR2"
  }

  def signalHandler(signum, frame):
    if signum in [signal.SIGINT, signal.SIGTERM]:
      print
      print "%s caught. Exiting." % signalToNameMap[signum]
      sys.exit(0)
    else:
      print >> sys.stderr
      print >> sys.stderr, "%s caught. Exiting." % signalToNameMap[signum]
      sys.exit(1)

  for each in signalToNameMap.keys():
    signal.signal(each, signalHandler)

def parseCommandLineArgs(args):
  (options, extra) = getopt.getopt(args[1:], "dp:h", ["debug", "port=",
    "help"])

  port = DEFAULT_PORT
  isDebugOutputEnabled = False

  for eachOptName, eachOptValue in options:
    if eachOptName in ("-d", "--debug"):
      isDebugOutputEnabled = True
    elif eachOptName in ("-p", "--port"):
      port = int(eachOptValue)
    elif eachOptName in ("-h", "--help"):
      printUsage(sys.stdout)
      sys.exit(0)

  if len(extra) != 1:
    message = "missing command line argument"
    if len(extra) > 1: message = "unexpected command line argument"
    print >> sys.stderr, "Error: %s" % message
    printUsage(sys.stderr)
    sys.exit(1)

  return (extra[0], port, isDebugOutputEnabled)

def printUsage(outStream):
  print >> outStream, """\
Usage: %s [-dph] HOSTNAME

Options:
  -d,--debug        Print debug messages to stderr
  -p,--port=NUMBER  Connect to port NUMBER on HOSTNAME (default: %d)
  -h,--help         Print usage message and exit\
""" % (getProgramName(), DEFAULT_PORT)


class TextStreamToPhiPosTranslator:
  def __init__(self, inputStream):
    self.inputStream = inputStream

  def readPhiPosition(self):
    while True:
      each = self.inputStream.readline()
      if each == "": raise Exception("EOF encountered on input stream")
      position = self.parseStreamLine(each)
      if position != None: return position

  def parseStreamLine(self, aString):
    parts = aString.strip().split(None, 3)
    if len(parts) == 0: return None
    try: return long(parts[-1])
    except Exception: return None


class ConnectionSession:
  def __init__(self, hostname, port, debugOutputEnabled):
    self.hostname = hostname
    self.port = port
    self.debugOutputEnabled = debugOutputEnabled
    self.inputTerminator = "\006"
    self.outputTerminator = "\r"
    self.stdinToPhiPosTranslator = TextStreamToPhiPosTranslator(sys.stdin)
    self.clientSocket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    self.clientSocket.connect((hostname, port))

  def go(self):
    while True:
      self.writePhiPosToPmacSim(self.readPhiPosFromStdin())

  def readPhiPosFromStdin(self):
    return self.stdinToPhiPosTranslator.readPhiPosition()

  def writePhiPosToPmacSim(self, position):
    self.sendCommand("PhiPos=%d" % position)

  def sendCommand(self, commandString):
    data = commandString + self.outputTerminator
    self.clientSocket.sendall(data)
    if self.debugOutputEnabled:
      print >> sys.stderr, "> \"%s\"" % data.encode("string_escape")
    reply = self.clientSocket.recv(128)
    if self.debugOutputEnabled:
      print >> sys.stderr, "< \"%s\"" % reply.encode("string_escape")
    if reply != self.inputTerminator:
      print >> sys.stderr, "error: unexpected reply from server: \"%s\"" % \
        reply.encode("string_escape")

if __name__ == '__main__':
  try:
    initializeSignalHandlers()
    main(sys.argv)
  except Exception, e:
    if isinstance(e, SystemExit):
      raise e
    else:
      print >> sys.stderr
      print >> sys.stderr, "Error: %s" % e
      sys.exit(1)
